/*
 * Time-stamp: <2004-10-19 22:45:49 poser>
 *
 * This program computes a number of measures of dynamism in
 * F0 contours. The F0 contour should consist of values in
 * Hertz as binary single precision floating point numbers.
 *	 
 * Copyright (C) 1991-2003 William J. Poser (billposer@alum.mit.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or go to the web page:  http://www.gnu.org/licenses/gpl.txt.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/file.h>
#include <math.h>
#include <string.h>
#include <getopt.h>

#define VERSION "2.2"

#define INITITEMS 256	/* Number of items for initial storage allocation. */
#define TRUE 1
#define FALSE 0
#define ERROR -1
#define SUCCESS 1

#define oddp(n) (n % 2)

char **xav;	/* External argument list */
char progname[]="pflux";
char version[]=VERSION "[" __DATE__ " " __TIME__ "]";

main(int ac,char *av[])
{
   float *f0;		/* Storage for F0 contour */
   float *sf0;		/* Sorted sub-array */
   float *cf0;		/* Compressed F0 contour */
   float *flucts;	/* List of fluctuation magnitudes */ 
   long pts;		/* Number of F0 estimates */
   long CompressedPts; /* Number of points in compressed contour */
   char *InFileName = NULL; /* Name of input file */
   long BytesNeeded;	/* Storage needed for sort */
   int skip;		/* Number of points to skip at each end */
   long i;
   int Debug = 0;
   int BadOpt= 0;
   int c;		/* Getopt option flag */

   double Delta = 0.0;	/* Threshold for considering adjacent F0 values different */

   double Median;

   double AverageRate;	/* Mean rate of F0 change across region */
   double AverageAbsRate; /* Mean rate of absolute F0 change */
   
   double Mean;
   double Max;
   double Min;
   double Range;
   double Variance;
   double StdDeviation;
   
   /* The usual, but for middle 95% of values */

   double Min95;
   double Max95;
   double Range95;
   double Mean95;
   double Variance95;
   double StdDeviation95;
	
   long FluctCnt;      /* Number of fluctuations */
   long FluctCnt15;    /* Number of fluctuations  greater than 15 Hz */
   long FluctCnt10Pct; /* Number of fluctuations greater than 10% of Range95 */
   long FluctCnt20Pct; /* Number of fluctuations greater than 20% of Range95 */
   long FluctCnt30Pct; /* Number of fluctuations greater than 30% of Range95 */
   long FluctCnt40Pct; /* Number of fluctuations greater than 40% of Range95 */
   long FluctCnt50Pct; /* Number of fluctuations greater than 50% of Range95 */
   long FluctCnt60Pct; /* Number of fluctuations greater than 60% of Range95 */
   long FluctCnt70Pct; /* Number of fluctuations greater than 70% of Range95 */
   long FluctCnt80Pct; /* Number of fluctuations greater than 80% of Range95 */
   long FluctCnt90Pct; /* Number of fluctuations greater than 90% of Range95 */
   long FluctCnt100Pct; /* Number of fluctuations greater than Range95 */

   double FluctMean;   /* Mean magnitude of fluctuations */
   double FluctVar;	/* Variance of fluctuations */
   
   double FloatMean(float *,long,long);
   double FloatVariance(float *,long,long);
   double AvgRate(float *,long);
   double AvgAbsRate(float *,long);
   long CountFlucts(float *,long,double,float *);
   int ReadBinaryFloats(char *FileName,long *Pts,float **Array,int InitItems);
   void IdentifySelf(FILE *);
   char *TimeString(void); 

   /*
   extern int optind;
   extern char *optarg;
   extern int getopt(int, char ** const, const char *);
   */

   /* Executable statements begin here */
   
   xav = av;
   
   while( (c = getopt(ac,av,"dt:")) != EOF){
     switch(c){
     case 'd':
       Debug = 1;
       break;
     case 't':
       Delta = atof(optarg);
       break;
     default:
       BadOpt=1;
     break;
     }
   }


   if(ac > optind){
     if(strcmp(av[optind],"-") == 0) InFileName=NULL;
     else InFileName = av[optind];
   }
   else BadOpt = TRUE;

   if(BadOpt){
     IdentifySelf(stderr);
     fprintf(stderr,"Program to compute measures of dynamism of F0 contours.\n");
     fprintf(stderr,"F0 contours should be Hz as single precision floats.\n");
     fprintf(stderr,"Output is written on the standard output.\n");
     fprintf(stderr,"If the debugging flag is given, the compressed F0 contour\n");
     fprintf(stderr,"used internally and the fluctuation list are also written.\n");
     fprintf(stderr,"If the input file is \"-\" stdin is read.\n");
     fprintf(stderr,"The threshold, 0.0 by default, is the difference between adjacent\nF0 values required for them to be considered different.\n");
     fprintf(stderr,"Usage: pflux (-d[ebug]) (-t <threshold>) <F0 contour file name>\n\n");
     exit(1);
   }


   /* Read and store F0 contour */

   if(ReadBinaryFloats(InFileName,&pts,&f0,(int)INITITEMS) == ERROR) exit(1);
   if(pts == 0L){
      fprintf(stdout,"%s: input contains no data - aborting.\n",av[0]);
      exit(0);
   }
   

   /* Make a sorted copy of f0 contour */

   BytesNeeded = (long)(pts * sizeof(float));
   sf0 = (float *) malloc( (size_t) BytesNeeded);
   if(sf0 == (float *) 0){
      fprintf(stderr,"Cannot allocate storage.\n"); 
      exit(1);
   }
   (void)memcpy((void *) sf0,(const void *) f0,(size_t) BytesNeeded);
   fshellsort(sf0,pts);


   /* Compute statistics */
   

   AverageRate = AvgRate(f0,pts);
   AverageAbsRate = AvgAbsRate(f0,pts);

   Mean = FloatMean(f0,0L,pts-1L);
   Max = sf0[pts-1L];
   Min = sf0[0L];
   Range = Max - Min;
   Variance = FloatVariance(f0,0,pts-1L);
   StdDeviation = sqrt(Variance);

   /*
    * Compute the median, which is the midpoint of the sorted array
    * If there are an even number of points in the array,
    * the median is the mean of the two central points.
    */

   if(oddp(pts)) Median = sf0[pts/2L];
   else Median = (sf0[pts/2L -1L] + sf0[pts/2L]) / 2.0;
   
   /* Compute 95% mid range statistics */
   
   skip = (int) (0.5 + ((double) pts * 0.025));
   if(skip > 0){
	   Min95 = sf0[skip-1];
	   Max95 = sf0[pts-skip-1];
	   Range95 = Max95 - Min95; 
	   Mean95 = FloatMean(sf0,skip-1,pts-skip-1);
	   Variance95 = FloatVariance(sf0,skip-1,pts-skip-1);
	   StdDeviation95 = sqrt(Variance95);
   }
   else{
	   Min95 = Min;
	   Max95 = Max;
	   Range95 = Range;
	   Mean95 = Mean;
	   Variance95 = Variance;
	   StdDeviation95 = StdDeviation;
   }


   /*	The sorted list is no longer required so we free it here */
	
   free( (void *) sf0);


   /* Make a compressed copy of array, with no levels */

   BytesNeeded = pts * sizeof(float);
   cf0 = (float *) malloc( (size_t) BytesNeeded);
   if(cf0 == (float *) 0){
	   fprintf(stderr,"Cannot allocate storage.\n"); 
      exit(1);
   }


   cf0[0] = f0[0];
   CompressedPts = 1L;
   for(i = 1L; i < pts; i++){
	if(fabs(f0[i] - f0[i-1]) >= Delta) cf0[CompressedPts++] = f0[i];
   }

   if(Debug)  DumpCompF0(cf0,CompressedPts);

   /* Allocate storage for fluctuation list */

   BytesNeeded = CompressedPts * sizeof(float);
   flucts = (float *) malloc( (size_t) BytesNeeded);
   if(flucts == (float *) 0){
	   fprintf(stderr,"Cannot allocate storage.\n"); 
      exit(1);
   }

   /* Compute fluctuation statistics */

   FluctCnt = CountFlucts(cf0,CompressedPts,0.0,flucts);
   if(Debug) (void)DumpFlucts(flucts,FluctCnt);

   FluctMean = FloatMean(flucts,0L,(long)FluctCnt-1L);
   FluctVar  = FloatVariance(flucts,0L,(long)FluctCnt-1L);
   FluctCnt15 = CountFlucts(cf0,CompressedPts,15.0,flucts);
   FluctCnt10Pct = CountFlucts(cf0,CompressedPts,0.1 * Range95,flucts);
   FluctCnt20Pct = CountFlucts(cf0,CompressedPts,0.2 * Range95,flucts);
   FluctCnt30Pct = CountFlucts(cf0,CompressedPts,0.3 * Range95,flucts);
   FluctCnt40Pct = CountFlucts(cf0,CompressedPts,0.4 * Range95,flucts);
   FluctCnt50Pct = CountFlucts(cf0,CompressedPts,0.5 * Range95,flucts);
   FluctCnt60Pct = CountFlucts(cf0,CompressedPts,0.6 * Range95,flucts);
   FluctCnt70Pct = CountFlucts(cf0,CompressedPts,0.7 * Range95,flucts);
   FluctCnt80Pct = CountFlucts(cf0,CompressedPts,0.8 * Range95,flucts);
   FluctCnt90Pct = CountFlucts(cf0,CompressedPts,0.9 * Range95,flucts);
   FluctCnt100Pct = CountFlucts(cf0,CompressedPts,1.0 * Range95,flucts);
   
   /* Write out report */

   printf("pflux version 2 - report generated %s\n",TimeString());
   printf("File %s    %d points\n\n",InFileName,pts);

   printf("AverageRate     %8.2f\n",AverageRate);
   printf("AverageAbsRate  %8.2f\n",AverageAbsRate);
   printf("FluctCnt        %8d\n",FluctCnt);
   printf("FluctMean       %8.2f\n",FluctMean);
   printf("FluctVar        %8.2f\n",FluctVar);
   printf("FluctCnt15      %8d\n",FluctCnt15);
   printf("FluctCnt10Pct   %8d\n",FluctCnt10Pct);
   printf("FluctCnt20Pct   %8d\n",FluctCnt20Pct);
   printf("FluctCnt30Pct   %8d\n",FluctCnt30Pct);
   printf("FluctCnt40Pct   %8d\n",FluctCnt40Pct);
   printf("FluctCnt50Pct   %8d\n",FluctCnt50Pct);
   printf("FluctCnt60Pct   %8d\n",FluctCnt60Pct);
   printf("FluctCnt70Pct   %8d\n",FluctCnt70Pct);
   printf("FluctCnt80Pct   %8d\n",FluctCnt80Pct);
   printf("FluctCnt90Pct   %8d\n",FluctCnt90Pct);
   printf("FluctCnt100Pct  %8d\n\n",FluctCnt100Pct);

   printf("Median          %8.2f\n",Median);

   printf("\n                Full Range     Mid 95%%\n\n");
   
   printf("Mean            %8.2f     %8.2f\n",Mean,Mean95);
   printf("Max             %8.2f     %8.2f\n",Max,Max95);
   printf("Min             %8.2f     %8.2f\n",Min,Min95);
   printf("Max-Min         %8.2f     %8.2f\n",Range,Range95); 
   printf("Max-Mean        %8.2f     %8.2f\n",Max-Mean,Max95-Mean95); 
   printf("Max-Median      %8.2f     %8.2f\n",Max-Median,Max95-Median); 
   printf("Variance        %8.2f     %8.2f\n",Variance,Variance95);
   printf("StdDeviation    %8.2f     %8.2f\n",StdDeviation,StdDeviation95);

   /* Clean up and exit */

   free( (void *) f0);
   free( (void *) cf0);
   free( (void *) flucts);
	
   exit(0);
}


/*
 * Read floating point numbers from a file 
 * or standard input, allocating the necessary storage.
 * Errors are reported on stderr.
 */	  

int
ReadBinaryFloats(
char *FileName,	/* Name of file from which to read - null means stdin */
long *Pts,	/* Return parameter - number of items read */
float **Array,	/* Return parameter - pointer to allocated storage */
int InitItems) /* Initial number of items for which to allocate storage */
{
   int infd;	/* Input file descriptor */
   long i = 0L;	/* Items read */
   float *fvals = NULL;	/* Storage */
   unsigned int NewBytes;	/* Bytes requested */
   size_t BytesAvailable = 0;
   size_t OldBytesAvailable;
   int BytesRead;
   
   /*
    *	Round InitItems up to be a multiple of the size of a float,
    * in order to ensure that each chunk of bytes read will contain
    * an integral number of floats.	(a/b & a/c => a/(b-c) ).
    */
   
   while (InitItems % sizeof(float) != 0) InitItems *= 2;
   
   
   /* Set up input stream */

   if(FileName == NULL) infd = fileno(stdin);
   else{
      infd = open(FileName,O_RDONLY);
      if(infd < 0){
	 fprintf(stderr,"%s: cannot open file %s.\n",
		 xav[0],FileName);
	 return(ERROR);
      }
   }
   
   while(1){
      /* Allocate storage if necessary */
      OldBytesAvailable = BytesAvailable;
      if(BytesAvailable == 0){
	 BytesAvailable = (size_t) (sizeof(float) * InitItems);
      }
      else BytesAvailable *= 2;
      NewBytes = (int) (BytesAvailable - OldBytesAvailable);
      fvals = (float *) realloc( (void *) fvals,BytesAvailable);
      if(fvals == NULL){
	 fprintf(stderr,"%s: ran out of memory.\n",xav[0]);
	 return(ERROR);
      }
      
      BytesRead = read(infd,(char *)(fvals + i),NewBytes);
      
      if(BytesRead < 0){
	 fprintf(stderr,"%s: error on read from %s%s.\n",
		 xav[0],
		 (FileName ? "file " : "std"),
		 (FileName ? FileName : "in"));
	 return (ERROR);
      }
      
      i += (long) (BytesRead/sizeof(float));
      if(BytesRead < NewBytes) break;
   }	/* End of while loop */
   
   *Pts = i;
   *Array = fvals;
   if(infd != fileno(stdin)) close(infd);
   return(SUCCESS);
}



double FloatMean(float *array,long BeginPoint,long EndPoint) 
{
   long i;
   long cnt;
   double sum = 0.0;

   cnt = (EndPoint - BeginPoint) + 1;

   if(cnt == 0L) return 0.0;

   for(i = BeginPoint; i <= EndPoint; i++){
      sum += (double) (array[i]);
   }

   return(sum/(double)cnt);
}



/* 
 * A shellsort for floating point values.
 */


fshellsort(v,n)
float *v;
long n;
{
   long gap;
   long i;
   long j;
   long temp;
   
   for (gap = n/2L; gap > 0L; gap /= 2L){
      for (i = gap; i < n; i++){
	 for (j = i-gap; j >=0L; j -=gap) {
	    if(v[j] <= v[j+gap]) break;
	    else{		/* Swap */
	       temp = v[j];
	       v[j] = v[j+gap];
	       v[j+gap] = temp;
	    }
	 }
      }
   }
}

double
FloatVariance(float *array,long first,long last)
{
	long i;
	long N;

	double Mean;
	double Deviation;
	double SquaredDeviations;

	N = last - first + 1L;
	if(N == 1L) return(0.0);

	Mean = FloatMean(array,first,last);
	SquaredDeviations = 0.0;

	for (i = first; i <= last; i++){
	   Deviation = (double)(array[i]) - Mean;
	   SquaredDeviations += (Deviation * Deviation);
	}

	return(SquaredDeviations / (double) (N - 1L));
}

/* Compute average rate of change */

double
AvgRate(float *vals,long pts)
{	
	if(pts == 0L) return((double) 0.0);
	else return((double) (vals[0] - vals[pts-1])/(double)pts);
}


/* Compute average absolute rate of change */

double
AvgAbsRate(float *vals,long pts)
{
	long i;
	double sum;

	if(pts == 0L) return(0.0);

	sum = 0.0;
	for(i=0L;i < pts-2L; i++){
		sum += fabs(vals[i] - vals[i+1]);
	}
	return(sum/(double)pts);
}


/*
 * Return the number of fluctuations of given magnitude,
 * leaving a list in the passed array.	 
 */

long
CountFlucts(float *cf,long cfcnt,double t,float *FluctList)
{
   long i;
   long fcnt = 0L;
   double delta;
   long PreviousIPt = 0L;

   for(i = 1L; i < cfcnt-1; i++){
      if( ( (cf[i] - cf[i-1]) * (cf[i+1] - cf[i]) ) < 0.0){ /* inflection point */
	 delta = fabs((double)(cf[i] - cf[PreviousIPt]));
	 if(delta >= t){
	    PreviousIPt = i;
	    FluctList[fcnt++] = (float) delta;
	 }
      }
   }

   /* The last point is an inflection point */

   delta = fabs((double)(cf[cfcnt-1] - cf[PreviousIPt]));
   if(delta >= t) FluctList[fcnt++] = (float) delta;
	
   return(fcnt);
}


int DumpFlucts(float *flcts, long cnt)
{
   long i;
   printf("\n");
   for(i=0L;i<cnt;i++) printf("<fluct %3d> %4.2f\n",i,flcts[i]);
   printf("\n");
}

int DumpCompF0(float *cf, long cnt)
{
   long i;
   printf("\nCompressed F0 Contour\n");
   for(i=0L;i<cnt;i++) printf("<%3d> %4.2f\n",i,cf[i]);
   printf("\n");
}

void
IdentifySelf(FILE *fp)   
{
  fprintf(fp,"%s - version %s\n",progname,version);
}

#include <time.h>
char *TimeString(void)
{
   time_t t;
   t = time(NULL);
   return(ctime(&t));
}
