/* Time-stamp: <2004-06-03 22:14:06 poser> */
/*
 * This program distorts a speech file by
 * inverting the sign of randomly chosen samples, where the
 * probability of inversion is 0.5, the algorithm used by Schroeder and Samuel.
 * The result is to flatten the spectrum to white noise and wipe out
 * any periodicity, thereby eliminating the F0 contour, while preserving
 * the amplitude envelope.
 *
 * Copyright (C) 1993-2004 William J. Poser (billposer@alum.mit.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or go to the web page:  http://www.gnu.org/licenses/gpl.txt.
 */


#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <sndfile.h>

#define BUFFER_SAMPLES 8192

static char pgname[] = "samuel";
static char version[]="2.0";
static char compdate[]=__DATE__ " " __TIME__ ;

#define RANDSEED	1066

#ifdef HPUX
#define srandom(x) srand(x)
#define RMAX	0x7fff	/* maximum value for  rand() */
#define frand() ((double)rand()/RMAX)
#endif

#ifdef BSD4_3
#define RMAX ((double) 0x7FFFFFFF)	/* 2^31 -1 */
#define frand() ((double)random()/RMAX)	/* Range is [0,1] */
#endif

#ifdef LINUX
#define frand() ((double)random()/RAND_MAX)	/* Range is [0,1] */
#endif


/*
 * This subroutine inverts the sign of randomly chosen samples
 * in the buffer, where the probability of inversion is 0.5.
 */

static void
Samuel(short * buf, long samples)
{
  long i;
   
  for(i = 0L;  i < samples; i++){
    if(frand() > 0.5) buf[i] = (- buf[i]);
  }
}

void
Version(void){
  fprintf(stderr,"%s %s \n",pgname,version);
  fprintf(stderr,"Compiled %s.\n",compdate);
  fprintf(stderr,"Copyright 2004 William J. Poser\n");
  fprintf(stderr,"Released under the terms of the GNU General Public License.\n");
  fprintf(stderr,"Report bugs to: billposer@alum.mit.edu.\n");
  putc('\n',stderr);
}

static void
Usage(void)
{
  fprintf(stderr,"\nProgram to distort speech by inverting random samples.\n");
  fprintf(stderr,"Usage: %s <input file> <output file>\n",pgname);
  putc('\n',stderr);
}

int
main(int ac, char **av){
   
  SNDFILE *infd;		/* Input file descriptor */
  SNDFILE *outfd;		/* Output file descriptor */
  int SamplesRead;			/* Bytes read */
  int SamplesWritten;			/* Bytes written */
  struct SF_INFO sinfo;
  int c;

  short spbuf[BUFFER_SAMPLES];

  /* Process command line arguments */

  opterr = 0;			/* Handle errors ourselves */
  while ( (c = getopt(ac,av,"hv")) != (-1)){
    switch(c)
      {
      case 'h':
	Usage();
	exit(1);
	break;
      case 'v':
	Version();
	exit(1);
	break;
      default:
	fprintf(stderr,"%s: illegal option flag %c\n",pgname,optopt);
	exit(3);
      }
  }
   
  if(ac < 3){
    Usage();
    exit(1);
  }

  /* Open input file */

  sinfo.format = 0;
  infd = sf_open (av[1],SFM_READ,&sinfo);
  if(infd == NULL){
    fprintf(stderr,"%s: could not open file %s for read.\n",pgname,av[1]);
    exit(2);
  }
  if(sinfo.channels != 1){
    fprintf(stderr,
	    "%s: audio file %s has %d channels. Only mono data can be processed.\n",
	    pgname,av[1],sinfo.channels);
    exit(3);
  }
   
  /* Open output file */
   
  outfd = sf_open(av[2],SFM_WRITE,&sinfo);
  if(outfd == NULL){
    fprintf(stderr,"%s: could not open file %s for write.\n",pgname,av[2]);
    exit(2);
  }
   
  srandom(RANDSEED);

  /* Work through input file, transforming and writing out */
   
  while(SamplesRead = sf_read_short(infd,spbuf,BUFFER_SAMPLES)){
    if(SamplesRead < 0){
      fprintf(stderr,"%s: error reading input file %s.\n",
	      av[0],
	      av[1]);
      unlink(av[2]);
      exit(1);
    }
    Samuel(spbuf,SamplesRead);
    SamplesWritten = sf_write_short(outfd,spbuf,SamplesRead);
    if(SamplesWritten != SamplesRead){
      fprintf(stderr,"%s: error writing output file %s.\n",
	      pgname,av[2]);
      unlink(av[2]);
      exit(4);
    }
  }
   
   /* Clean up and exit */
   
  sf_close(infd);
  sf_close(outfd);
  exit(0);
}

