/* Time-stamp: <2004-10-01 22:49:55 poser> */
/*
 * Add sound files, which must have the same sampling rate and
 * number of channels.
 * The file format of the output file is determined by its extension.
 * The number of channels, sampling rate, and data format of the output
 * file will be those of the first input file.
 *
 * Copyright (C) 2004 William J. Poser (billposer@alum.mit.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or go to the web page:  http://www.gnu.org/licenses/gpl.txt.
 */

/* Figure out whether to restrict operation to single channel or if not
 * how to handle multiple channels */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <sndfile.h>

static char pgname[] = "sndadd";
static char version[]="1.0";
static char compdate[]=__DATE__ " " __TIME__ ;

void
Version(void){
  fprintf(stderr,"%s %s \n",pgname,version);
  fprintf(stderr,"Compiled %s.\n",compdate);
  fprintf(stderr,"Copyright 2004 William J. Poser\n");
  fprintf(stderr,"Released under the terms of the GNU General Public License.\n");
  fprintf(stderr,"Report bugs to: billposer@alum.mit.edu.\n");
  putc('\n',stderr);
}

void
Usage(void){
  fprintf(stderr,"Add sound files, which must have the\n");
  fprintf(stderr,"same sampling rate and number of channels.\n");
  fprintf(stderr,"The file format of the output file is determined by its extension.\n");
  fprintf(stderr,"The number of channels, sampling rate, and data format of the output\n");
  fprintf(stderr,"file will be those of the first input file.\n");
  fprintf(stderr,"Usage:\t%s -o <output file> <input file>+\n",pgname);
  fprintf(stderr,"\t%s -h \t\t print this help message\n",pgname);
  fprintf(stderr,"\t%s -v \t\t print version informationn\n",pgname);
  putc('\n',stderr);
}


main(int ac, char **av)
{
  SNDFILE *infd;		/* Input file descriptor */
  SNDFILE *outfd;		/* Output file descriptor */
  struct SF_INFO InputInfo;
  struct SF_INFO StandardInfo;
  char *InFileName;
  char *OutFileName = NULL;
  int c;
  long Frames;
  long SamplesRead;	
  long SamplesWritten;	
  double *ABuffer;
  double *IBuffer;
  long i;
  double Min;
  double Max;
  double Range;
  double ScaleFactor;
  double ScaledMin;
  double Shift;

   
  /* Process command line arguments */

  opterr = 0;			/* Handle errors ourselves */
  while ( (c = getopt(ac,av,"ho:v")) != (-1)){
    switch(c)
      {
      case 'h':
	Usage();
	exit(1);
	break;
      case 'v':
	Version();
	exit(1);
	break;
      case 'o':
	OutFileName = optarg;
	break;
      default:
	fprintf(stderr,"%s: illegal option flag %c\n",pgname,optopt);
	exit(3);
      }
  }

  if(ac < 4){
    Usage();
    exit(2);
  }

  if(OutFileName == NULL){
    fprintf(stderr,"%s: no output file specified.\n",pgname);
    Usage();
    exit(2);
  }

  /* Open the first input file so as to determine what format we are using
     and the number of samples  */

  InFileName = av[optind];
  StandardInfo.format = 0;
  infd = sf_open(InFileName,SFM_READ,&StandardInfo);
  if(infd == NULL){
    fprintf(stderr,"%s: could not open file %s for read.\n",pgname,InFileName);
    exit(2);
  }
  Frames = StandardInfo.frames;
  if(StandardInfo.channels != 1){
    fprintf(stderr,"Only single channel data is supported.\n");
    exit(1);
  }
  
  /* Open output file */
   
  outfd = sf_open(OutFileName,SFM_WRITE,&StandardInfo);
  if (outfd == NULL){
    fprintf(stderr,"%s: could not open file %s for write.\n",pgname,OutFileName);
    exit(2);
  }

  /* Allocate the buffers */

  ABuffer = (double *) malloc((size_t)(Frames * sizeof(double)));
  if (ABuffer == NULL){
    fprintf(stderr,"Could not allocate accumulator buffer.\n");
    exit(1);
  }	

  IBuffer = (double *) malloc((size_t)(Frames * sizeof(double)));
  if (IBuffer == NULL){
    fprintf(stderr,"Could not allocate input buffer.\n");
    exit(1);
  }	

  /* Read the data from the first input file into the accumulator buffer */

 SamplesRead = sf_read_double(infd,ABuffer,Frames);
 if(sf_error(infd)){
      fprintf(stderr,"Error reading file %s:\n",InFileName);
      fprintf(stderr,"\t%s\n",sf_strerror(infd));
      exit(1);
 }
 sf_close(infd);
 optind++;


 /* Loop over remaining input files, adding their contents to accumulator */

  while (optind < ac){
    InFileName = av[optind];
    InputInfo.format = 0;
    infd = sf_open(InFileName,SFM_READ,&InputInfo);
    if(infd == NULL){
      fprintf(stderr,"%s: could not open file %s for read.\n",pgname,InFileName);
      exit(2);
    }
    if(InputInfo.channels != StandardInfo.channels){
      fprintf(stderr,"Input file %s: incorrect number of channels = %d. Should be %d.\n",
	      InFileName,
	      InputInfo.channels,StandardInfo.channels);
      unlink(OutFileName);
      exit(2);
    }
    if(InputInfo.samplerate != StandardInfo.samplerate){
      fprintf(stderr,"Input file %s: incorrect sampling rate = %d. Should be %d.\n",
	      InFileName,
	      InputInfo.samplerate,StandardInfo.samplerate);
      unlink(OutFileName);
      exit(2);
    }
    if(InputInfo.frames != Frames){
      fprintf(stderr,"Input file %s: incorrect frame count %ld\n",
	InFileName,InputInfo.frames);
      exit(1);
    }

     SamplesRead = sf_read_double(infd,IBuffer,Frames);
     if( (SamplesRead != Frames) || sf_error(infd)){
      fprintf(stderr,"Error reading file %s:\n",InFileName);
      fprintf(stderr,"\t%s\n",sf_strerror(infd));
      exit(1);
     }
    sf_close(infd);
    
    /* Add the contents of the current input buffer to the accumulator */
    for(i=0L;i<Frames;i++){
      ABuffer[i] += IBuffer[i];
    }

    optind++;
  }

  free((void *) IBuffer);


  /* Normalize sum to range [-1.0,1.0] */

  Max = Min = ABuffer[0];
  for(i=1;i<Frames;i++){
    if(ABuffer[i] > Max) Max = ABuffer[i];
    if(ABuffer[i] < Min) Min = ABuffer[i];
  }
  Range = Max-Min;
  ScaleFactor = 2.0/Range;
  ScaledMin = Min * ScaleFactor;
  Shift = -1.0 * (ScaledMin + 1.0);

  for(i=0;i<Frames;i++){
    ABuffer[i] = (ABuffer[i] *ScaleFactor) + Shift;
  } 	

  /* Write out result */

  SamplesWritten = sf_write_double(outfd,ABuffer,Frames);
   if( (SamplesWritten != Frames) || (sf_error(outfd))){
      fprintf(stderr,"Error writing file %s:\n",OutFileName);
      fprintf(stderr,"\t%s\n",sf_strerror(outfd));
      exit(2);
    }

  sf_close(outfd);
  exit(0);
}
