/* Time-stamp: <2004-10-01 13:41:26 wjposer> */
/*
 * Concatenate sound files, which must have the same sampling rate and
 * number of channels.
 * The file format of the output file is determined by its extension.
 * The number of channels, sampling rate, and data format of the output
 * file will be those of the first input file.
 *
 * Copyright (C) 1985-2004 William J. Poser (billposer@alum.mit.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or go to the web page:  http://www.gnu.org/licenses/gpl.txt.
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <sndfile.h>

#define NO_ERROR 0
#define ERROR (-1)

static char pgname[] = "sndcat";
static char version[]="1.0";
static char compdate[]=__DATE__ " " __TIME__ ;

void
Version(void){
  fprintf(stderr,"%s %s \n",pgname,version);
  fprintf(stderr,"Compiled %s.\n",compdate);
  fprintf(stderr,"Copyright 2004 William J. Poser\n");
  fprintf(stderr,"Released under the terms of the GNU General Public License.\n");
  fprintf(stderr,"Report bugs to: billposer@alum.mit.edu.\n");
  putc('\n',stderr);
}

void
Usage(void){
  fprintf(stderr,"Concatenate sound files, which must have the\n");
  fprintf(stderr,"same sampling rate and number of channels.\n");
  fprintf(stderr,"The file format of the output file is determined by its extension.\n");
  fprintf(stderr,"The number of channels, sampling rate, and data format of the output\n");
  fprintf(stderr,"file will be those of the first input file.\n");
  fprintf(stderr,"Usage:\t%s -o <output file> <input file>+\n",pgname);
  fprintf(stderr,"\t%s -h \t\t print this help message\n",pgname);
  fprintf(stderr,"\t%s -v \t\t print version informationn\n",pgname);
  putc('\n',stderr);
}

/* 
 * Copy audio data from input to output until input is exhausted.
 * Return code indicating error or no error.
 */

#define SBUFSIZE 8192

int
CopySound(
  SNDFILE *infd,
  SNDFILE *outfd,
  char *InputFile,
  char *OutFileName)
{
  long SamplesRead;	
  long SamplesWritten;	
  double sbuf[SBUFSIZE];

  while( (SamplesRead = sf_read_double(infd,sbuf,SBUFSIZE)) > 0){
    if(sf_error(infd)){
      fprintf(stderr,"Error reading file %s:\n",InputFile);
      fprintf(stderr,"\t%s\n",sf_strerror(infd));
      return(ERROR);
    }
    SamplesWritten = sf_write_double(outfd,sbuf,SamplesRead);
    if( (SamplesWritten != SamplesRead) || (sf_error(outfd))){
      fprintf(stderr,"Error writing file %s:\n",OutFileName);
      fprintf(stderr,"\t%s\n",sf_strerror(outfd));
      return(ERROR);
    }
  }
  return(NO_ERROR);
}

main(int ac, char **av)
{
  int InputCnt = 0;
  SNDFILE *infd;		/* Input file descriptor */
  SNDFILE *outfd;		/* Output file descriptor */
  struct SF_INFO InputInfo;
  struct SF_INFO StandardInfo;
  char *InFileName;
  char *OutFileName = NULL;
  int c;
   
  /* Process command line arguments */

  opterr = 0;			/* Handle errors ourselves */
  while ( (c = getopt(ac,av,"ho:v")) != (-1)){
    switch(c)
      {
      case 'h':
	Usage();
	exit(1);
	break;
      case 'v':
	Version();
	exit(1);
	break;
      case 'o':
	OutFileName = optarg;
	break;
      default:
	fprintf(stderr,"%s: illegal option flag %c\n",pgname,optopt);
	exit(3);
      }
  }

  if(ac < 4){
    Usage();
    exit(2);
  }

  if(OutFileName == NULL){
    fprintf(stderr,"%s: no output file specified.\n",pgname);
    exit(2);
  }

  /* Open the first input file so as to determine what format we are using */

  InFileName = av[optind];
  StandardInfo.format = 0;
  infd = sf_open(InFileName,SFM_READ,&StandardInfo);
  if(infd == NULL){
    fprintf(stderr,"%s: could not open file %s for read.\n",pgname,InFileName);
    exit(2);
  }
  
  /* Open output file */
   
  outfd = sf_open(OutFileName,SFM_WRITE,&StandardInfo);
  if (outfd == NULL){
    fprintf(stderr,"%s: could not open file %s for write.\n",pgname,OutFileName);
    exit(2);
  }

  /* Copy the data from the first input file */

  if(CopySound(infd,outfd,InFileName,OutFileName) == ERROR){
    unlink(OutFileName);
    exit(2);
  }
  sf_close(infd);
  optind++;

  while (optind < ac){
    InFileName = av[optind];
    InputInfo.format = 0;
    infd = sf_open(InFileName,SFM_READ,&InputInfo);
    if(infd == NULL){
      fprintf(stderr,"%s: could not open file %s for read.\n",pgname,InFileName);
      exit(2);
    }
    if(InputInfo.channels != StandardInfo.channels){
      fprintf(stderr,"Input file %s: incorrect number of channels = %d. Should be %d.\n",
	      InFileName,
	      InputInfo.channels,StandardInfo.channels);
      unlink(OutFileName);
      exit(2);
    }
    if(InputInfo.samplerate != StandardInfo.samplerate){
      fprintf(stderr,"Input file %s: incorrect sampling rate = %d. Should be %d.\n",
	      InFileName,
	      InputInfo.samplerate,StandardInfo.samplerate);
      unlink(OutFileName);
      exit(2);
    }
    if(CopySound(infd,outfd,InFileName,OutFileName) == ERROR) exit(2);
    sf_close(infd);
    optind++;
  }

  sf_close(outfd);
  exit(0);
}

